/* eslint-disable @typescript-eslint/no-unsafe-assignment */
import { Injectable } from '@nestjs/common';
import { PrismaService } from 'src/prisma/prisma.service';
import {
  GraphEdge,
  Graph,
  GraphId,
  GraphNode,
  NearestReturnType,
} from './navigation.entity';
@Injectable()
export class GraphService {
  constructor(private readonly prismaService: PrismaService) {}

  /**
   * Get a graph by its ID.
   * @param id The ID of the graph.
   * @returns The graph object or null if not found.
   */
  async getGraph(id: GraphId): Promise<Graph | null> {
    const graph = await this.prismaService.graph.findUnique({
      where: { id },
      include: {
        nodes: {
          include: {
            edgesA: true,
            edgesB: true,
          },
          orderBy: {
            id: 'asc',
          },
        },
      },
    });

    if (!graph) return null;

    // Map nodes to include both edgesA and edgesB (duplicates allowed)
    const nodes: GraphNode[] = graph.nodes.map((node) => {
      const allEdges: GraphEdge[] = [...node.edgesA, ...node.edgesB];

      return {
        id: node.id,
        label: node.label,
        x: node.x,
        y: node.y,
        z: node.z,
        graphId: node.graphId,
        edges: allEdges.map((e) => ({
          id: e.id,
          nodeAId: e.nodeAId,
          nodeBId: e.nodeBId,
        })),
      };
    });

    return {
      id: graph.id,
      originX: graph.OriginX,
      originY: graph.OriginY,
      originZ: graph.OriginZ,
      scale: graph.scale,
      nodes,
    };
  }

  /**
   * Get a node by its ID.
   * @param id The ID of the node.
   * @returns The graph node object or null if not found.
   */
  async getNode(id: string): Promise<GraphNode | null> {
    const node = await this.prismaService.node.findUnique({
      where: { id },
      include: {
        edgesA: true,
        edgesB: true,
      },
    });

    if (!node) return null;

    const allEdges: GraphEdge[] = [...node.edgesA, ...node.edgesB];

    return {
      id: node.id,
      label: node.label,
      x: node.x,
      y: node.y,
      z: node.z,
      graphId: node.graphId,
      edges: allEdges.map((e) => ({
        id: e.id,
        nodeAId: e.nodeAId,
        nodeBId: e.nodeBId,
      })),
    };
  }

  /**
   * Get the nearest node to the given coordinates.
   * @param x The x-coordinate.
   * @param y The y-coordinate.
   * @param z The z-coordinate.
   * @param yThreshold The optional y-coordinate threshold for filtering nodes.
   * @param returnType The type of return value (shallow or deep).
   * @returns The ID of the nearest node, the nearest node object, or null if no nodes exist.
   */
  async getNearestNode(
    x: number,
    y: number,
    z: number,
    yThreshold?: number,
    returnType: NearestReturnType = NearestReturnType.SHALLOW,
    graphId?: string, // Optional graphId parameter
  ): Promise<string | GraphNode | null> {
    // Find nodes scoped to the specified graphId (if provided)
    const nodes = await this.prismaService.node.findMany({
      where: graphId ? { graphId } : undefined,
    });

    if (!nodes || nodes.length === 0) return null;

    // Filter nodes by yThreshold if provided
    const filteredNodes =
      yThreshold !== undefined
        ? nodes.filter((node) => Math.abs(node.y - y) <= yThreshold)
        : nodes;

    if (!filteredNodes.length) return null;

    // Find nearest node by Euclidean distance (x, y, z)
    let minDist = Number.POSITIVE_INFINITY;
    let nearest: (typeof nodes)[0] | null = null;

    for (const node of filteredNodes) {
      const dist = Math.sqrt(
        Math.pow(node.x - x, 2) +
          Math.pow(node.y - y, 2) +
          Math.pow(node.z - z, 2),
      );
      if (dist < minDist) {
        minDist = dist;
        nearest = node;
      }
    }

    if (!nearest) return null;

    if (returnType === NearestReturnType.SHALLOW) {
      return String(nearest.id);
    } else {
      // Use getNode to include edges
      return await this.getNode(String(nearest.id));
    }
  }
}
